<?php
/**
 * Plugin Name: One Code Studio – Admin Notes
 * Plugin URI: https://onecodestudio.com/plugins/admin-notes
 * Description: Private admin notes for your team. Safe Mode + Quick Add anchor + Edit at top + Native drag-and-drop (manual save) + Multi-select delete.
 * Version: 0.7.1
 * Author: One Code Studio
 * Author URI: https://onecodestudio.com
 * License: GPLv2 or later
 * Text Domain: ocs-admin-notes
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

register_activation_hook(__FILE__, function(){
    global $wp_version;
    if ( version_compare(PHP_VERSION, '7.0', '<') ) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die('One Code Studio – Admin Notes requires PHP 7.0+.');
    }
    if ( isset($wp_version) && version_compare($wp_version, '5.2', '<') ) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die('One Code Studio – Admin Notes requires WordPress 5.2+.');
    }
});

class OCS_Admin_Notes_Safe {
    const OPTION = 'ocs_admin_notes';
    const NONCE  = 'ocs_admin_notes_nonce';
    const CAP    = 'edit_posts';
    const VER    = '0.7.1';

    private $labels = array(
        ''          => array('General', '#9aa0a6', '#ffffff'),
        'urgent'    => array('Urgent',  '#d93025', '#ffffff'),
        'followup'  => array('Follow-up','#f29900', '#1f2937'),
        'info'      => array('Info',    '#1a73e8', '#ffffff'),
        'done'      => array('Done',    '#10b981', '#032b24'),
        'idea'      => array('Idea',    '#8b5cf6', '#ffffff'),
    );

    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'));
        add_action('admin_head', array($this, 'print_inline_css'));
        add_action('admin_post_ocs_add_note', array($this, 'handle_add_note'));
        add_action('admin_post_ocs_delete_note', array($this, 'handle_delete_note'));
        add_action('admin_post_ocs_export_csv', array($this, 'handle_export_csv'));
        add_action('admin_post_ocs_update_note', array($this, 'handle_update_note'));
        add_action('admin_post_ocs_save_order', array($this, 'handle_save_order'));
        add_action('admin_post_ocs_bulk_delete', array($this, 'handle_bulk_delete'));
        add_action('admin_bar_menu', array($this, 'add_admin_bar_link'), 100);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
    }

    private function can_use() {
        return is_user_logged_in() && current_user_can(self::CAP);
    }

    private function get_notes() {
        $notes = get_option(self::OPTION, array());
        if (!is_array($notes)) { $notes = array(); }
        return $notes;
    }

    private function save_notes($notes) {
        update_option(self::OPTION, array_values($notes));
    }

    private function new_id() {
        return 'n_' . wp_generate_password(6, false, false);
    }

    private function normalize_notes($notes) {
        $order = 10;
        foreach ($notes as $i => $n) {
            if (!is_array($n)) $n = array();
            if (empty($n['id'])) $n['id'] = $this->new_id();
            if (!isset($n['label']) || !array_key_exists($n['label'], $this->labels)) $n['label'] = '';
            if (!isset($n['order']) || !is_numeric($n['order'])) { $n['order'] = $order; $order += 10; }
            if (!isset($n['text'])) $n['text'] = '';
            if (!isset($n['user'])) $n['user'] = '';
            if (!isset($n['time'])) $n['time'] = current_time('mysql');
            $notes[$i] = $n;
        }
        usort($notes, function($a,$b){ return intval($a['order']) - intval($b['order']); });
        return $notes;
    }

    public function print_inline_css() {
        if ( !is_admin() || !$this->can_use() ) return;
        echo '<style id="ocs-admin-notes-css">
.wrap .ocs-card{max-width:900px;padding:12px;background:#fff;border:1px solid #e5e7eb;border-radius:12px;margin-bottom:14px}
.ocs-notes{margin:0;padding:0}
.ocs-note{list-style:none;margin:0 0 .75em 0;padding:.75em;border:1px solid #e5e7eb;border-radius:8px;background:#fff;display:grid;grid-template-columns:auto 1fr;gap:8px;align-items:start}
.ocs-note .ocs-select{margin-top:4px}
.ocs-pill{display:inline-block;padding:.15em .6em;border-radius:999px;font-size:11px;font-weight:600;margin-right:8px}
.ocs-actions a{margin-right:10px}
.ocs-drag-hint{opacity:.8;border:1px dashed #cbd5e1}
.ocs-drag-handle{cursor:grab;display:inline-block;margin-right:8px;color:#64748b}
.ocs-toolbar{display:flex;gap:8px;align-items:center;flex-wrap:wrap;margin:.5rem 0 1rem}
.ocs-edit-panel{border:1px solid #93c5fd;background:#eff6ff;border-radius:12px;padding:12px;margin:10px 0}
.ocs-select-all{display:flex;gap:8px;align-items:center;margin:.5rem 0}
        </style>';
    }

    public function add_menu_page() {
        if (!$this->can_use()) return;
        add_menu_page(
            esc_html__('Admin Notes','ocs-admin-notes'),
            esc_html__('Admin Notes','ocs-admin-notes'),
            self::CAP,
            'ocs-admin-notes',
            array($this, 'render_screen'),
            'dashicons-edit',
            3
        );
    }

    public function add_admin_bar_link($wp_admin_bar) {
        if (!is_admin_bar_showing() || !$this->can_use()) return;
        $url = admin_url('admin.php?page=ocs-admin-notes#ocs-quick-add');
        $wp_admin_bar->add_node(array(
            'id'    => 'ocs-admin-notes-quick-add',
            'title' => __('➕ Quick Admin Note','ocs-admin-notes'),
            'href'  => $url,
            'meta'  => array('title' => __('Add a quick admin note','ocs-admin-notes')),
        ));
    }

    public function enqueue_assets($hook) {
        if ( !$this->can_use() ) return;
        if ($hook !== 'toplevel_page_ocs-admin-notes') return;
        wp_enqueue_script('ocs-admin-notes-dnd', plugins_url('assets/dnd.js', __FILE__), array(), self::VER, true);
        wp_localize_script('ocs-admin-notes-dnd', 'OCS_ADMIN_NOTES_DND', array(
            'listId' => 'ocs-notes',
            'dragClass' => 'ocs-drag-hint'
        ));
    }

    private function label_pill($slug) {
        $def = isset($this->labels[$slug]) ? $this->labels[$slug] : $this->labels[''];
        $bg = $def[1]; $fg = $def[2];
        return '<span class="ocs-pill" style="background:'.$bg.';color:'.$fg.'">'.$def[0].'</span>';
    }

    private function render_add_form() {
        echo '<div id="ocs-quick-add" class="ocs-card"><form method="post" action="'.esc_url( wp_nonce_url( admin_url('admin-post.php?action=ocs_add_note'), self::NONCE ) ).'">';
        echo '<textarea name="ocs_note" rows="3" style="width:100%" placeholder="'.esc_attr__('Type a note...','ocs-admin-notes').'"></textarea>';
        echo '<p><label>'.esc_html__('Label:','ocs-admin-notes').' ';
        echo '<select name="ocs_label">';
        foreach ($this->labels as $slug => $def) {
            echo '<option value="'.esc_attr($slug).'">'.esc_html($def[0]).'</option>';
        }
        echo '</select></label> ';
        echo '<button type="submit" class="button button-primary">'.esc_html__('Save Note','ocs-admin-notes').'</button></p>';
        echo '</form></div>';
    }

    public function render_screen() {
        echo '<div class="wrap"><h1>'.esc_html__('Admin Notes','ocs-admin-notes').'</h1>';

        // Edit-at-top panel
        $edit_id = isset($_GET['edit']) ? sanitize_text_field($_GET['edit']) : '';
        if ($edit_id) {
            $notes = $this->normalize_notes($this->get_notes());
            $idx = -1;
            foreach ($notes as $i=>$note) { if (isset($note['id']) && $note['id']===$edit_id) { $idx=$i; break; } }
            if ($idx >= 0) {
                $note = $notes[$idx];
                echo '<div id="ocs-edit" class="ocs-edit-panel">';
                echo '<h2>'.esc_html__('Edit Note','ocs-admin-notes').'</h2>';
                echo '<form method="post" action="'.esc_url( wp_nonce_url( admin_url('admin-post.php?action=ocs_update_note'), self::NONCE ) ).'">';
                echo '<input type="hidden" name="id" value="'.esc_attr($note['id']).'"/>';
                echo '<textarea name="ocs_note" rows="4" style="width:100%" autofocus>'.esc_textarea($note['text']).'</textarea>';
                echo '<p><label>'.esc_html__('Label:','ocs-admin-notes').' ';
                echo '<select name="ocs_label">';
                foreach ($this->labels as $slug => $def) {
                    $sel = selected($slug, (isset($note['label']) ? $note['label'] : ''), false);
                    echo '<option value="'.esc_attr($slug).'" '.$sel.'>'.esc_html($def[0]).'</option>';
                }
                echo '</select></label> ';
                echo '<button type="submit" class="button button-primary">'.esc_html__('Update Note','ocs-admin-notes').'</button> ';
                $cancel = esc_url( admin_url('admin.php?page=ocs-admin-notes') );
                echo '<a class="button" href="'.$cancel.'">'.esc_html__('Cancel','ocs-admin-notes').'</a></p>';
                echo '</form></div>';
                echo '<script>location.hash = "#ocs-edit";</script>';
            }
        }

        echo '<div class="ocs-toolbar">';
        echo '<a class="button" href="'.esc_url( admin_url('admin.php?page=ocs-admin-notes#ocs-quick-add') ).'">'.esc_html__('➕ Quick Add','ocs-admin-notes').'</a> ';
        echo '<a class="button" href="'.esc_url( wp_nonce_url( admin_url('admin-post.php?action=ocs_export_csv'), self::NONCE ) ).'">'.esc_html__('Export CSV','ocs-admin-notes').'</a>';
        echo '</div>';

        // Quick Add
        $this->render_add_form();

        $notes = $this->normalize_notes($this->get_notes());
        if (empty($notes)) {
            echo '<p>'.esc_html__('No notes yet.','ocs-admin-notes').'</p></div>';
            return;
        }

        // Reorder + Bulk Delete combined form
        $save_action = wp_nonce_url( admin_url('admin-post.php?action=ocs_save_order'), self::NONCE );
        $bulk_action = wp_nonce_url( admin_url('admin-post.php?action=ocs_bulk_delete'), self::NONCE );
        echo '<form method="post" action="'.esc_url($save_action).'">';
        echo '<div class="ocs-select-all"><label><input type="checkbox" id="ocs_sel_all"/> '.esc_html__('Select all','ocs-admin-notes').'</label></div>';
        echo '<input type="hidden" id="ocs_order" name="ocs_order" value="" />';
        echo '<p><em>'.esc_html__('Tip: Drag items, then click "Save Order". Check boxes to delete multiple at once.','ocs-admin-notes').'</em></p>';
        echo '<ul id="ocs-notes" class="ocs-notes">';
        foreach ($notes as $n) {
            $id = esc_attr($n['id']);
            $text = isset($n['text']) ? $n['text'] : '';
            $user = isset($n['user']) ? $n['user'] : '';
            $time = isset($n['time']) ? $n['time'] : '';
            $edited = isset($n['edited']) ? $n['edited'] : '';
            $label = isset($n['label']) ? $n['label'] : '';
            $del = esc_url( wp_nonce_url( admin_url('admin-post.php?action=ocs_delete_note&id='.$id), self::NONCE ) );
            $edit_url = esc_url( add_query_arg(array('page'=>'ocs-admin-notes','edit'=>$id), admin_url('admin.php')) . '#ocs-edit' );
            echo '<li class="ocs-note" draggable="true" data-id="'.$id.'">';
            echo '<div class="ocs-select"><input type="checkbox" name="sel[]" value="'.$id.'"/></div>';
            echo '<div>';
            echo '<span class="ocs-drag-handle" title="'.esc_attr__('Drag to reorder','ocs-admin-notes').'">↕</span>';
            echo $this->label_pill($label);
            echo '<div>'.wpautop( esc_html( $text ) ).'</div>';
            $meta = sprintf(__('By %1$s on %2$s','ocs-admin-notes'), $user, $time);
            if ($edited) $meta .= ' • ' . sprintf(__('Edited %s','ocs-admin-notes'), esc_html($edited));
            echo '<small>'.esc_html($meta).'</small>';
            echo '<div class="ocs-actions"><a href="'.$edit_url.'">'.esc_html__('Edit','ocs-admin-notes').'</a> <a href="'.$del.'" onclick="return confirm(\'' . esc_js(__('Delete this note?','ocs-admin-notes')) . '\');">'.esc_html__('Delete','ocs-admin-notes').'</a></div>';
            echo '</div>';
            echo '</li>';
        }
        echo '</ul>';
        echo '<p>';
        echo '<button type="submit" class="button button-secondary">'.esc_html__('Save Order','ocs-admin-notes').'</button> ';
        echo '<button type="submit" class="button button-link-delete" formaction="'.esc_url($bulk_action).'" onclick="return confirm(\''.esc_js(__('Delete selected notes?','ocs-admin-notes')).'\');">'.esc_html__('Delete Selected','ocs-admin-notes').'</button>';
        echo '</p>';
        echo '</form>';

        // Clean inline JS for select-all (no extra escaping), bind click + change
        echo '<script>
        (function(){
          var all = document.getElementById("ocs_sel_all");
          if(!all) return;
          function toggle(){
            var boxes = document.querySelectorAll("#ocs-notes input[type=checkbox][name=\'sel[]\']");
            boxes.forEach(function(cb){ cb.checked = all.checked; });
          }
          all.addEventListener("click", toggle);
          all.addEventListener("change", toggle);
        })();
        </script>';

        echo '</div>';
    }

    /* --------- Handlers --------- */
    public function handle_add_note() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $text  = trim( (string) ( isset($_POST['ocs_note']) ? $_POST['ocs_note'] : '' ) );
        $label = sanitize_text_field( isset($_POST['ocs_label']) ? $_POST['ocs_label'] : '' );
        if (!array_key_exists($label, $this->labels)) $label = '';
        if ($text !== '') {
            $notes = $this->normalize_notes($this->get_notes());
            $maxOrder = 0;
            foreach ($notes as $n) { $maxOrder = max($maxOrder, intval(isset($n['order']) ? $n['order'] : 0)); }
            $notes[] = array(
                'id'    => $this->new_id(),
                'text'  => wp_kses_post( wp_unslash($text) ),
                'user'  => function_exists('wp_get_current_user') ? wp_get_current_user()->user_login : 'user',
                'time'  => current_time('mysql'),
                'label' => $label,
                'order' => $maxOrder + 10,
            );
            $this->save_notes($notes);
        }
        wp_safe_redirect( admin_url('admin.php?page=ocs-admin-notes') );
        exit;
    }

    public function handle_update_note() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $id    = sanitize_text_field(isset($_POST['id']) ? $_POST['id'] : '');
        $text  = trim( (string) ( isset($_POST['ocs_note']) ? $_POST['ocs_note'] : '' ) );
        $label = sanitize_text_field( isset($_POST['ocs_label']) ? $_POST['ocs_label'] : '' );
        if (!array_key_exists($label, $this->labels)) $label = '';

        $notes = $this->normalize_notes($this->get_notes());
        $idx = -1;
        foreach ($notes as $i=>$n) { if (isset($n['id']) && $n['id'] === $id) { $idx = $i; break; } }
        if ($idx >= 0 && $text !== '') {
            $notes[$idx]['text']   = wp_kses_post( wp_unslash($text) );
            $notes[$idx]['label']  = $label;
            $notes[$idx]['edited'] = current_time('mysql');
            $this->save_notes($notes);
        }
        wp_safe_redirect( admin_url('admin.php?page=ocs-admin-notes') );
        exit;
    }

    public function handle_delete_note() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $id = sanitize_text_field(isset($_GET['id']) ? $_GET['id'] : '');
        $notes = $this->get_notes();
        $notes = array_values(array_filter($notes, function($n) use ($id){
            return isset($n['id']) && $n['id'] !== $id;
        }));
        $this->save_notes($notes);
        wp_safe_redirect( admin_url('admin.php?page=ocs-admin-notes') );
        exit;
    }

    public function handle_bulk_delete() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $ids = isset($_POST['sel']) ? (array) $_POST['sel'] : array();
        $ids = array_map('sanitize_text_field', $ids);
        if (!empty($ids)) {
            $notes = $this->get_notes();
            $notes = array_values(array_filter($notes, function($n) use ($ids){
                return !(isset($n['id']) && in_array($n['id'], $ids, true));
            }));
            $this->save_notes($notes);
        }
        wp_safe_redirect( admin_url('admin.php?page=ocs-admin-notes') );
        exit;
    }

    public function handle_export_csv() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $notes = $this->normalize_notes($this->get_notes());
        $filename = 'ocs-admin-notes-' . date('Ymd-His') . '.csv';
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename='.$filename);
        header('Pragma: no-cache');
        header('Expires: 0');
        $out = fopen('php://output', 'w');
        fputcsv($out, array('id','text','user','time','label','order','edited'));
        foreach ($notes as $n) {
            $row = array(
                isset($n['id']) ? $n['id'] : '',
                isset($n['text']) ? wp_strip_all_tags($n['text']) : '',
                isset($n['user']) ? $n['user'] : '',
                isset($n['time']) ? $n['time'] : '',
                isset($n['label']) ? $n['label'] : '',
                isset($n['order']) ? $n['order'] : '',
                isset($n['edited']) ? $n['edited'] : '',
            );
            fputcsv($out, $row);
        }
        fclose($out);
        exit;
    }

    public function handle_save_order() {
        if ( !$this->can_use() ) wp_die(__('You do not have permission.','ocs-admin-notes'));
        if ( ! wp_verify_nonce( isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '', self::NONCE ) ) wp_die(__('Invalid nonce.','ocs-admin-notes'));
        $order_str = isset($_POST['ocs_order']) ? sanitize_text_field($_POST['ocs_order']) : '';
        $ids = array_filter(array_map('trim', explode(',', $order_str)));
        $notes = $this->normalize_notes($this->get_notes());
        $map = array(); foreach ($notes as $n) { if (isset($n['id'])) $map[$n['id']] = $n; }
        $order = 10; $sorted = array();
        foreach ($ids as $id) { if (isset($map[$id])) { $n=$map[$id]; $n['order']=$order; $order+=10; $sorted[]=$n; unset($map[$id]); } }
        foreach ($map as $n) { $n['order']=$order; $order+=10; $sorted[]=$n; }
        $this->save_notes($sorted);
        wp_safe_redirect( admin_url('admin.php?page=ocs-admin-notes') );
        exit;
    }
}

new OCS_Admin_Notes_Safe();
