<?php
namespace Focus_Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

class Settings {
    private static $instance = null;
    private $option_key = 'focus_admin_defaults';
    private $usermeta_key = 'focus_admin_userprefs';

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', [ $this, 'register_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_post_focus_admin_save_prefs', [ $this, 'handle_save' ] );
    }

    public function register_settings_page() {
        add_options_page(
            __( 'Focus Admin', 'focus-admin' ),
            __( 'Focus Admin', 'focus-admin' ),
            'manage_options',
            'focus-admin',
            [ $this, 'render_settings_page' ]
        );
    }

    public function register_settings() {
        register_setting( 'focus_admin', $this->option_key );
    }

    private function get_all_menu_slugs() {
        // Build a list of current top-level menu slugs/titles for selection UI
        global $menu;
        $items = [];
        foreach ( (array) $menu as $m ) {
            if ( empty( $m[2] ) || empty( $m[0] ) ) continue;
            $title = wp_strip_all_tags( $m[0] );
            $items[ $m[2] ] = $title;
        }
        ksort( $items );
        return $items;
    }

    public function handle_save() {
        if ( ! current_user_can( 'read' ) ) {
            wp_die( 'Unauthorized' );
        }
        check_admin_referer( 'focus_admin_save' );

        $user_id = get_current_user_id();
        $prefs = [
            'keep_menus'   => isset( $_POST['keep_menus'] ) ? array_map( 'sanitize_text_field', (array) $_POST['keep_menus'] ) : [],
            'hide_notices' => isset( $_POST['hide_notices'] ) ? (bool) $_POST['hide_notices'] : false,
            'hide_toolbar' => isset( $_POST['hide_toolbar'] ) ? (bool) $_POST['hide_toolbar'] : false,
        ];
        update_user_meta( $user_id, $this->usermeta_key, $prefs );

        wp_safe_redirect( add_query_arg( [ 'page' => 'focus-admin', 'updated' => '1' ], admin_url( 'options-general.php' ) ) );
        exit;
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'read' ) ) return;

        $user_id = get_current_user_id();
        $prefs   = get_user_meta( $user_id, $this->usermeta_key, true );
        if ( ! is_array( $prefs ) ) {
            $prefs = [
                'keep_menus'   => [ 'index.php', 'edit.php', 'upload.php', 'edit.php?post_type=page', 'profile.php' ],
                'hide_notices' => true,
                'hide_toolbar' => true,
            ];
        }

        $menus = $this->get_all_menu_slugs();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Focus Admin Preferences', 'focus-admin' ); ?></h1>
            <?php if ( isset( $_GET['updated'] ) ): ?>
                <div class="notice notice-success is-dismissible"><p><?php esc_html_e('Preferences saved.', 'focus-admin'); ?></p></div>
            <?php endif; ?>
            <p><?php esc_html_e( 'Choose what stays visible in Focus Mode. These settings apply only to your account.', 'focus-admin' ); ?></p>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'focus_admin_save' ); ?>
                <input type="hidden" name="action" value="focus_admin_save_prefs" />

                <h2 class="title"><?php esc_html_e('Keep These Menu Items', 'focus-admin'); ?></h2>
                <fieldset style="columns: 2; -moz-columns: 2;">
                    <?php foreach ( $menus as $slug => $title ) : ?>
                        <label style="display:block;margin-bottom:6px;">
                            <input type="checkbox" name="keep_menus[]" value="<?php echo esc_attr( $slug ); ?>"
                                <?php checked( in_array( $slug, $prefs['keep_menus'], true ) ); ?> />
                            <?php echo esc_html( $title . ' (' . $slug . ')' ); ?>
                        </label>
                    <?php endforeach; ?>
                </fieldset>

                <h2 class="title"><?php esc_html_e('Other Options', 'focus-admin'); ?></h2>
                <label style="display:block;margin:6px 0;">
                    <input type="checkbox" name="hide_notices" value="1" <?php checked( ! empty( $prefs['hide_notices'] ) ); ?> />
                    <?php esc_html_e('Hide admin notices in Focus Mode (you can reveal them with the Notices button).', 'focus-admin'); ?>
                </label>
                <label style="display:block;margin:6px 0;">
                    <input type="checkbox" name="hide_toolbar" value="1" <?php checked( ! empty( $prefs['hide_toolbar'] ) ); ?> />
                    <?php esc_html_e('Hide toolbar items (WP logo, comments) in Focus Mode.', 'focus-admin'); ?>
                </label>

                <?php submit_button( __( 'Save Preferences', 'focus-admin' ) ); ?>
            </form>

            <hr />
            <p><strong><?php esc_html_e('Tip:', 'focus-admin'); ?></strong> <?php esc_html_e('Use the admin bar button “Focus: ON/OFF” to toggle quickly.', 'focus-admin'); ?></p>
        </div>
        <?php
    }
}
