<?php
namespace Focus_Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

class Core {
    private static $instance = null;
    private $usermeta_key = 'focus_admin_userprefs';
    private $toggle_key   = 'focus_admin_toggle';

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Toggle handler (via admin_init)
        add_action( 'admin_init', [ $this, 'maybe_handle_toggle' ] );

        // Admin bar toggle button
        add_action( 'admin_bar_menu', [ $this, 'add_adminbar_toggle' ], 1000 );

        // Apply focus mode filters
        add_action( 'admin_menu', [ $this, 'filter_admin_menu' ], 999 );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_filter( 'admin_body_class', [ $this, 'body_class' ] );

        // Toolbar declutter
        add_action( 'admin_bar_menu', [ $this, 'declutter_toolbar' ], 999 );
    }

    public function is_focus_mode_on( $user_id = 0 ) {
        $user_id = $user_id ? $user_id : get_current_user_id();
        $on = get_user_meta( $user_id, $this->toggle_key, true );
        return (bool) $on;
    }

    public function set_focus_mode( $on ) {
        update_user_meta( get_current_user_id(), $this->toggle_key, $on ? 1 : 0 );
    }

    public function maybe_handle_toggle() {
        if ( ! is_admin() ) return;

        if ( isset($_GET['focus_admin_toggle']) ) {
            check_admin_referer( 'focus_admin_toggle' );
            $current = $this->is_focus_mode_on();
            $this->set_focus_mode( ! $current );
            // redirect back (without query args)
            $url = remove_query_arg( [ 'focus_admin_toggle', '_wpnonce' ] );
            wp_safe_redirect( $url ? $url : admin_url() );
            exit;
        }
    }

    public function add_adminbar_toggle( $wp_admin_bar ) {
        if ( ! is_user_logged_in() || ! current_user_can( 'read' ) ) return;
        if ( is_network_admin() ) return;

        $on = $this->is_focus_mode_on();
        $title = $on ? 'Focus: ON' : 'Focus: OFF';
        $args = [
            'id'    => 'focus-admin-toggle',
            'title' => $title,
            'href'  => wp_nonce_url( add_query_arg( 'focus_admin_toggle', '1', admin_url() ), 'focus_admin_toggle' ),
            'meta'  => [ 'class' => 'focus-admin-toggle' ],
        ];
        $wp_admin_bar->add_node( $args );
    }

    public function get_user_prefs( $user_id = 0 ) {
        $user_id = $user_id ? $user_id : get_current_user_id();
        $defaults = [
            'keep_menus'    => [ 'index.php', 'edit.php', 'upload.php', 'edit.php?post_type=page', 'profile.php' ],
            'hide_notices'  => true,
            'hide_toolbar'  => true,
        ];
        $prefs = get_user_meta( $user_id, $this->usermeta_key, true );
        if ( ! is_array( $prefs ) ) $prefs = [];
        return wp_parse_args( $prefs, $defaults );
    }

    public function enqueue_assets( $hook ) {
        // Enqueue admin CSS/JS only
        wp_enqueue_style( 'focus-admin', FOCUS_ADMIN_URL . 'assets/css/admin.css', [], FOCUS_ADMIN_VERSION );
        wp_enqueue_script( 'focus-admin', FOCUS_ADMIN_URL . 'assets/js/admin.js', [ 'jquery' ], FOCUS_ADMIN_VERSION, true );
        wp_localize_script( 'focus-admin', 'FocusAdmin', [
            'focusOn' => $this->is_focus_mode_on(),
        ] );
    }

    public function body_class( $classes ) {
        if ( $this->is_focus_mode_on() ) {
            $classes .= ' focus-admin--on';
        }
        return $classes;
    }

    public function filter_admin_menu() {
        if ( ! $this->is_focus_mode_on() ) return;

        $prefs = $this->get_user_prefs();
        if ( empty( $prefs['keep_menus'] ) || ! is_array( $prefs['keep_menus'] ) ) return;

        global $menu, $submenu;
        $keep = array_map( 'strval', $prefs['keep_menus'] );

        // Iterate over $menu items and remove those not in keep list.
        foreach ( $menu as $index => $m ) {
            $slug = isset( $m[2] ) ? $m[2] : '';
            if ( $slug && ! in_array( $slug, $keep, true ) ) {
                remove_menu_page( $slug );
            }
        }

        // Optionally prune some common submenus (keep all for MVP).
        // Future: allow per-submenu control.
    }

    public function declutter_toolbar( $wp_admin_bar ) {
        if ( ! $this->is_focus_mode_on() ) return;

        $prefs = $this->get_user_prefs();
        if ( ! empty( $prefs['hide_toolbar'] ) ) {
            // Remove WP logo and comments (common clutter)
            $wp_admin_bar->remove_node( 'wp-logo' );
            $wp_admin_bar->remove_node( 'comments' );
            // You can remove more nodes here if desired.
        }
    }
}
